<?php

namespace Modules\Flowmaker\Listeners;

use App\Models\Company;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;
use Modules\Flowmaker\Models\Flow;
use Modules\Flowmaker\Models\ContactState;
use Modules\Wpbox\Models\Reply;

class RespondOnMessage
{

    public function handleMessageByContact($event){
        try {
            $contact = $event->message->contact;
            $message = $event->message;
            
            if ($contact->enabled_ai_bot && !$message->bot_has_replied) {
                $this->routeMessage($contact, $message);
            }
        } catch (\Throwable $th) {
            Log::error('Flow routing error', ['error' => $th->getMessage()]);
        }
    }

    private function routeMessage($contact, $message): void
    {
        // Load candidate flows ordered by priority (desc)
        $flows = Flow::where('company_id', $contact->company_id)
                    ->where('is_active', true)
                    ->orderBy('priority', 'desc')
                    ->get();

        Log::info('Checking flows for keyword match', [
            'contact_id' => $contact->id,
            'message' => $message->value,
            'total_flows' => $flows->count(),
            'flows' => $flows->pluck('id', 'name')->toArray()
        ]);

        $currentFlowId = $contact->getCurrentFlowId();

        // 1) Preemptive keyword routing: always check for a keyword match first
        $matchedFlow = null;
        foreach ($flows as $flow) {
            if ($this->hasKeywordMatch($message, $flow)) {
                $matchedFlow = $flow;
                break;
            }
        }

        if ($matchedFlow) {
            // If switching from another flow, clear previous flow's state
            if ($currentFlowId && $currentFlowId !== $matchedFlow->id) {
                $this->clearFlowState($contact->id, $currentFlowId);
                Log::info('Preempting current flow due to keyword match', [
                    'contact_id' => $contact->id,
                    'from_flow_id' => $currentFlowId,
                    'to_flow_id' => $matchedFlow->id,
                    'to_flow_name' => $matchedFlow->name
                ]);
            }

            // Set new current flow if needed
            if (!$currentFlowId || $currentFlowId !== $matchedFlow->id) {
                $contact->setCurrentFlowId($matchedFlow->id);
            }

            $matchedFlow->processMessage($message);
            return;
        }

        // 2) Continue current flow if active and no keyword match occurred
        if ($currentFlowId) {
            $flow = Flow::find($currentFlowId);
            if ($flow && $flow->isActive()) {
                Log::info('Continuing current flow (no keyword preemption)', [
                    'contact_id' => $contact->id,
                    'flow_id' => $flow->id,
                    'flow_name' => $flow->name
                ]);
                $flow->processMessage($message);
                return;
            }
        }

        // 3) Fallback logic: First try incoming_message flows, then priority 0 flows
        $incomingMessageFlow = null;
        $priorityZeroFlow = null;
        
        foreach ($flows as $flow) {
            // Check for incoming_message trigger
            $flowData = is_string($flow->flow_data) ? json_decode($flow->flow_data, true) : $flow->flow_data;
            if ($flowData && isset($flowData['nodes'])) {
                foreach ($flowData['nodes'] as $node) {
                    if ($node['type'] === 'incoming_message') {
                        $incomingMessageFlow = $flow;
                        break 2;
                    }
                }
            }
            
            // Also check for priority 0 flows as backup
            if ($flow->priority == 0) {
                $priorityZeroFlow = $flow;
            }
        }
        
        // Debug: Log all flows with their priorities
        Log::info('All flows with priorities', [
            'flows' => $flows->map(function($flow) {
                return [
                    'id' => $flow->id,
                    'name' => $flow->name,
                    'priority' => $flow->priority
                ];
            })->toArray()
        ]);
        
        if ($incomingMessageFlow) {
            Log::info('Using incoming_message flow as fallback', [
                'contact_id' => $contact->id,
                'flow_id' => $incomingMessageFlow->id,
                'flow_name' => $incomingMessageFlow->name,
                'priority' => $incomingMessageFlow->priority
            ]);
            $incomingMessageFlow->processMessage($message);
        } else if ($priorityZeroFlow) {
            Log::info('Using priority 0 flow as fallback', [
                'contact_id' => $contact->id,
                'flow_id' => $priorityZeroFlow->id,
                'flow_name' => $priorityZeroFlow->name,
                'priority' => $priorityZeroFlow->priority
            ]);
            $priorityZeroFlow->processMessage($message);
        } else {
            Log::warning('No fallback flow found', [
                'contact_id' => $contact->id,
                'company_id' => $contact->company_id
            ]);
        }
    }

    private function hasKeywordMatch($message, Flow $flow): bool
    {
        // Handle both string and array flow_data
        $flowData = is_string($flow->flow_data) ? json_decode($flow->flow_data, true) : $flow->flow_data;
        if (!$flowData || !isset($flowData['nodes'])) return false;
        
        foreach ($flowData['nodes'] as $node) {
            if ($node['type'] === 'keyword_trigger' && isset($node['data']['keywords'])) {
                foreach ($node['data']['keywords'] as $keyword) {
                    $keywordValue = strtolower($keyword['value']);
                    $messageValue = strtolower($message->value);
                    $matchType = $keyword['matchType'] ?? 'exact';
                    
                    $isMatch = false;
                    if ($matchType === 'exact') {
                        $isMatch = ($messageValue === $keywordValue);
                    } else if ($matchType === 'contains') {
                        $isMatch = str_contains($messageValue, $keywordValue);
                    }
                    
                    if ($isMatch) {
                        Log::info('Keyword match found', [
                            'flow_id' => $flow->id,
                            'flow_name' => $flow->name,
                            'keyword' => $keyword['value'],
                            'match_type' => $matchType,
                            'message' => $message->value
                        ]);
                        return true;
                    }
                }
            }
        }
        
        return false;
    }

    private function clearFlowState(int $contactId, int $flowId): void
    {
        // Clear all per-flow states (including current_node) for a clean switch
        ContactState::where('contact_id', $contactId)
            ->where('flow_id', $flowId)
            ->delete();
    }

    public function subscribe($events)
    {
        $events->listen(
            'Modules\Wpbox\Events\ContactReplies',
            [RespondOnMessage::class, 'handleMessageByContact']
        );
    }

}
